#ifndef __CImageManager__
#define __CImageManager__

//	===========================================================================

#include <Collections/TCountedPointerArray.hpp>
#include <Basics/CString.hpp>
#include "IImage.hpp"

using Exponent::Basics::CString;
using Exponent::Collections::TCountedPointerArray;
using Exponent::GUI::Graphics::IImage;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Graphics
		{
			/**
			 * @class CImageManager CImageManager.hpp
			 * @brief Manages image loaded and getting in an application static wide format\n
			 *
			 * CimageManager allows all the image loading for an application to happen in once nice tidy place.\n
			 * This class is heavily under construction and as such you are advised to *not use it yet!*. \n
			 * @todo Add XML reading
			 *
			 * @date 17/04/2006
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 * @version 1.0.1 Added getImageOnPath function
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CImageManager.hpp,v 1.8 2007/03/03 02:59:20 paul Exp $
			 */
			class CImageManager
			{

			public:

//	===========================================================================

				const static char CIMAGE_MANAGER_XML_TAG_ROOT[];							/**< The tag used for the root of an image xml file */
				const static char CIMAGE_MANAGER_XML_TAG_VERSION[];							/**< Version of the XML file */
				const static char CIMAGE_MANAGER_XML_TAG_IMAGES[];							/**< The tag used to start the images block */
				const static char CIMAGE_MANAGER_XML_TAG_ANIMATIONS[];						/**< The tag used to start the animations block */
				
				const static char CIMAGE_MANAGER_XML_TAG_CHILD_IMAGE[];						/**< The tag used to declare an image */
				const static char CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_RELATIVE_PATH[];		/**< The attribute name for the image relative path */
				const static char CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_FILENAME[];			/**< The attribute name for the image filename */
				const static char CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_NAME[];				/**< The attribute name for the image reference */

				const static char CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_FRAME_HEIGHT[];	/**< The attribute name fot the animations number of frames */
				const static char CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_NUM_FRAMES[];		/**< The attribute name fot the animations number of frames */

//	===========================================================================

				/**
				 * @class CImageReference CImageManager.hpp
				 * @brief Reference to an image, key pair of image and name
				 */
				class CImageReference : public CCountedObject
				{
					/** @cond */
					EXPONENT_CLASS_DECLARATION;
					/** @endcond */

				public:

//	===========================================================================

					/**
					 * Construction
					 */
					CImageReference();

					/**
					 * Construction
					 * @param image The image to store
					 * @param referenceName The name of the image
					 */
					CImageReference(IImage *image, const CString &referenceName);

					/**
					 * Destruction
					 */
					virtual ~CImageReference();

//	===========================================================================

					/**
					 * Set the image
					 * @param image The image to store
					 */
					void setImage(IImage *image) { EXCHANGE_COUNTED_OBJECTS(m_image, image); }

					/**
					 * Set the reference name
					 * @param referenceName The name of the image
					 */
					void setReferenceName(const CString &referenceName) { m_referenceName = referenceName; }

//	===========================================================================

					/**
					 * Get the image
					 * @retval IImage* The image
					 */
					IImage *getImage() { return m_image; }

					/**
					 * Get teh reference name
					 * @retval const CString& The name of the image
					 */
					const CString &getReferenceName() const { return m_referenceName; }

//	===========================================================================

				protected:

//	===========================================================================

					IImage *m_image;				/**< The image */
					CString m_referenceName;		/**< The name */
				};


				/**
				 * @class CAnimationReference CImageManager.hpp
				 * @brief Reference to an animation image, key pair of image and name along with sizes
				 * @see CImageReference
				 */
				class CAnimationReference: public CCountedObject
				{
					/** @cond */
					EXPONENT_CLASS_DECLARATION;
					/** @endcond */

//	===========================================================================

				public:

//	===========================================================================

					/**
					 * Construction
					 */
					CAnimationReference();

					/**
					 * Construction
					 * @param image The image to store
					 * @param referenceName The name of the image
					 * @param frameHeight The size of each frame
					 * @param numberOfFrames The number of frames in the animation
					 */
					CAnimationReference(IImage *image, const CString &referenceName, const long frameHeight, const long numberOfFrames);

					/**
					 * Destruction
					 */
					virtual ~CAnimationReference();

//	===========================================================================

					/**
					 * Set the image
					 * @param image The image to store
					 */
					void setImage(IImage *image) { m_image.setImage(image); }

					/**
					 * Set the reference name
					 * @param referenceName The name of the image
					 */
					void setReferenceName(const CString &referenceName) { m_image.setReferenceName(referenceName); }

					/**
					 * Set the sizes of the animation
					 * @param frameHeight The size of each frame
					 * @param numberOfFrames The number of frames in the animation
					 */
					void setAnimationSizes(const long frameHeight, const long numberOfFrames) { m_frameHeight = frameHeight; m_numberOfFrames = numberOfFrames; }

//	===========================================================================

					/**
					 * Get the image
					 * @retval IImage* The image
					 */
					IImage *getImage() { return m_image.getImage(); }

					/**
					 * Get teh reference name
					 * @retval const CString& The name of the image
					 */
					const CString &getReferenceName() const { return m_image.getReferenceName(); }

					/**
					 * Get the sizes of the animation
					 * @param frameHeight On return holds the size of each frame
					 * @param numberOfFrames On return holds the number of frames in the animation
					 */
					void getAnimationSizes(long &frameHeight, long &numberOfFrames) { frameHeight = m_frameHeight; numberOfFrames = m_numberOfFrames; }

//	===========================================================================

				protected:

//	===========================================================================

					CImageReference m_image;													/**< The image */
					long m_frameHeight;															/**< The height of each frame */
					long m_numberOfFrames;														/**< The total number of frames */
				};

//	===========================================================================

				/**
				 * Set the default path to images
				 * @param defaultPath The default path. All added images expect path parameter to be relative to tis position
				 */
				static void setDefaultPathToImages(const CSystemString &defaultPath);

				/**
				 * Get the default path
				 * @retval const CSystemString& The default image path
				 */
				static const CSystemString &getDefaultPathToImages() { return CIMAGE_MANAGER_DEFAULT_RESOURCE_PATH; }

//	===========================================================================

				/**
				 * Add an image to the manager
				 * @param path The folder that contains the image
				 * @param filename The name of the file to load. Currently only TGA 32bpp format is supported
				 * @param referenceName The name to refere to the image with, this must be unique
				 * @retval bool True if the image was loaded, false otherwise
				 */
				static bool addImage(const CSystemString &path, const CString &filename, const CString &referenceName);

				/**
				 * Add an animation image
				 * @param path The folder that contains the image
				 * @param filename The name of the file to load. Currently only TGA 32bpp format is supported
				 * @param referenceName The name to refere to the image with, this must be unique
				 * @param frameHeight The height of each frame
				 * @param numberOfFrames The number of frames in the image
				 * @retval bool True if the image was loaded, false otherwise
				 */
				static bool addAnimation(const CSystemString &path, const CString &filename, const CString &referenceName, const long frameHeight, const long numberOfFrames);

//	===========================================================================

				/**
				 * Load the images from a file\n
				 * The image manager expects the filename to point to a valid xml document. Each xml document should have (inside the root) two children, one for animations, one for images\n
				 * The images have 3 tags, The relative path, the filename and the reference name\n
				 * The animations have five attributes, the 3 above, a frame height (height of each frame in the image list) and a number of frames (total number of frames in animation)\n
				 * Here is example code to write a definition file\n\n
				 * @code
				 * // Add to a root node
				 * CXMLNode *root = new CXMLNode(CIMAGE_MANAGER_XML_TAG_ROOT, NULL);
				 * root->addAttribute(CIMAGE_MANAGER_XML_TAG_VERSION, "1.0.0");				// Currently only one defined. But make sure you include for future usage
				 *
				 * // One image and one animation node
				 * CXMLNode *images     = new CXMLNode(CIMAGE_MANAGER_XML_TAG_IMAGES,     root);
				 * CXMLNode *animations = new CXMLNode(CIMAGE_MANAGER_XML_TAG_ANIMATIONS, root);
				 *
				 * // Add our images
				 * CXMLNode *myImage = new CXMLNode(CIMAGE_MANAGER_XML_TAG_CHILD_IMAGE, images);
				 * myImage->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_RELATIVE_PATH,  "/Path/To/Place");
				 * myImage->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_FILENAME,  "MyImage.tga");
				 * myImage->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_NAME,  "MyImagesName");
				 *
				 * // Add our animations
				 * CXMLNode *myAnimation = new CXMLNode(CIMAGE_MANAGER_XML_TAG_CHILD_IMAGE, animations);
				 * myAnimation->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_RELATIVE_PATH, "/Path/To/Place");
				 * myAnimation->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_FILENAME, "MyImage.tga");
				 * myAnimation->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_NAME, "MyImagesName");
				 * myAnimation->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_FRAME_HEIGHT, "25");
				 * myAnimation->addAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_NUM_FRAMES, "127");
				 * @endcode
				 * @param filename The name of the file that contains the image definitons
				 * @retval bool True if loaded properly, false otherwise
				 * @note THE SYSTEM DESCRIBED BELOW IS NOW DEPRECATED (AND IS INCLUDED FOR BACKWARDS COMPATIBILITY ONLY!), USE THE XML LISTING AS ABOVE\n
				 * A file is expected to have two parts, animation and images. These parts\n
				 * can be in any order desired inside the file.\n
				 * After an identifier (either \<animation\> or \<images\>) each line after that must contain\n
				 * an image or an animation\n
				 * Images should be declared like this\n
				 * RelativePath, Filename, ReferenceName\n
				 * And animations should be declared like this\n
				 * RelativePath, Filename, ReferenceName, FrameHeight, NumberOfFrames\n
				 * lines with # symbols at the start are comments and are ignored\n
				 * Note that if an image is in the root folder, you can use {} for the relative path to signify this
				 */
				static bool loadFromDefinitionFile(const CSystemString &filename);

//	===========================================================================

				/**
				 * Get an image with a specific name
				 * @param referenceName The name of the image
				 * @retval IImage * The image requeseted or null on error
				 */
				static IImage *getImage(const CString &referenceName);

				/**
				 * Get a reference name for an image
				 * @param image The image
				 * @retval CString The reference name or CSTRING_EMPTY_STRING on error
				 */
				static CString getReferenceNameForImage(const IImage *image);

				/**
				 * Get an animation image with a specific name
				 * @param referenceName The name of the image
				 * @param frameHeight On return contains the height of each frame
				 * @param numberOfFrames On return contains the number of frames in the image
				 * @retval IImage * The image requeseted or null on error
				 */
				static IImage *getAnimation(const CString &referenceName, long &frameHeight, long &numberOfFrames);

				/**
				 * Get an animation image with a specific name
				 * @param referenceName The name of the image
				 * @retval CAnimationReference* The image requeseted or null on error
				 */
				static CImageManager::CAnimationReference *getAnimation(const CString &referenceName);

//	===========================================================================

				/**
				 * Remove all the items from the image library
				 */
				static void clearImageLibrary();

//	===========================================================================
				
				/**
				 * Get an image from a path
				 * @param path The path to the image
				 * @retval IImage* The image or NULL on error
				 */
				static IImage *getImageOnPath(const CSystemString &path);

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Read from an XML document 
				 * @param filename The name of the file to load
				 * @retval bool TRue if loaded, false otherwise
				 */
				static bool readFromXML(const CSystemString &filename);

//	===========================================================================

				static TCountedPointerArray<CImageReference> CIMAGE_MANAGER_IMAGES;				/**< The array of normal images */
				static TCountedPointerArray<CAnimationReference> CIMAGE_MANAGER_ANIMATIONS;		/**< The array of animation images */
				static CSystemString CIMAGE_MANAGER_DEFAULT_RESOURCE_PATH;						/**< Default resource path */
			};
		}
	}
}
#endif		// End of CImageManager.hpp